<?php

class FrmStrpActionsController extends FrmTransActionsController {


	/**
	 * Generate the HTML from Stripe rather than Formidable.
	 *
	 * @since 2.0
	 */
	public static function show_card_callback( $callback ) {
		return 'FrmStrpActionsController::show_card';
	}

	/**
	 * Override the credit card field HTML if there is a Stripe action.
	 *
	 * @since 2.0
	 */
	public static function show_card( $field, $field_name, $atts ) {
		$html_id = $atts['html_id'];

		if ( self::has_stripe_processed( $field['form_id'] ) && ! FrmFormsController::just_created_entry( $field['form_id'] ) ) {
			// Don't show the card field if the token has already been set.
			global $frm_vars;
			$has_error = isset( $frm_vars['trans_filled'] );
			if ( ! $has_error ) {
				echo '<div class="frm-card-errors frm_error" id="frm_error_' . esc_attr( $html_id ) . '" role="alert"></div>';
				return;
			}
		}

		$actions = self::get_actions_before_submit( $field['form_id'] );
		if ( empty( $actions ) && is_callable( 'FrmProCreditCardsController::show_in_form' ) ) {
			FrmProCreditCardsController::show_in_form( $field, $field_name, $atts );
		}

		$intents = array();

		include FrmStrpAppHelper::plugin_path() . '/views/payments/card-field.php';
	}

	/**
	 * @since 2.0
	 */
	public static function get_actions_before_submit( $form_id ) {
		$payment_actions = self::get_actions_for_form( $form_id );
		foreach ( $payment_actions as $k => $payment_action ) {
			$gateway = $payment_action->post_content['gateway'];
			$is_stripe = $gateway === 'stripe' || ( is_array( $gateway ) && in_array( 'stripe', $gateway ) );
			if ( ! $is_stripe || empty( $payment_action->post_content['amount'] ) ) {
				unset( $payment_actions[ $k ] );
			}
		}

		/*
		if ( count( $payment_actions > 1 ) ) {
			// TODO: How to pick? Check conditional logic.
			// Create payment intent for each one.
		}
		*/
		return $payment_actions;
	}

	/**
	 * If the names are being used on the CC fields,
	 * make sure it doesn't prevent the submission if Stripe has approved
	 */
	public static function remove_cc_validation( $errors, $field, $values ) {
		$has_processed = self::has_stripe_processed( $field->form_id ) || isset( $_POST[ 'frmintent' . $field->form_id ] );
		if ( FrmAppHelper::is_admin_page( 'formidable-entries' ) && isset( $_GET['frm_action'] ) && 'edit' === sanitize_text_field( $_GET['frm_action'] ) ) {
			$has_processed = true;
		}

		if ( ! $has_processed ) {
			return $errors;
		}

		$field_id = isset( $field->temp_id ) ? $field->temp_id : $field->id;
		if ( isset( $errors[ 'field' . $field_id . '-cc' ] ) ) {
			unset( $errors[ 'field' . $field_id . '-cc' ] );
		}
		if ( isset( $errors[ 'field' . $field_id ] ) ) {
			unset( $errors[ 'field' . $field_id ] );
		}

		return $errors;
	}

	public static function trigger_gateway( $action, $entry, $form ) {
		$response = array(
			'success'      => false,
			'run_triggers' => false,
			'show_errors'  => true,
		);
		$atts     = compact( 'action', 'entry', 'form' );

		$amount = self::prepare_amount( $action->post_content['amount'], $atts );
		if ( empty( $amount ) || $amount == 000 ) {
			$response['error'] = __( 'Please specify an amount for the payment', 'formidable-stripe' );
			return $response;
		}

		// retrieve the token generated by stripe.js
		if ( ! self::has_stripe_processed( $form->id ) ) {
			$response['error'] = __( 'The Stripe Token is missing.', 'formidable-stripe' );
			return $response;
		}

		if ( ! self::stripe_is_configured() ) {
			$response['error'] = __( 'There was a problem communicating with Stripe. Please try again.', 'formidable-stripe' );
			return $response;
		}

		$customer = self::set_customer_with_token( $atts );
		if ( ! is_object( $customer ) ) {
			$response['error'] = $customer;
			return $response;
		}

		// attempt to charge the customer's card
		if ( 'recurring' === $action->post_content['type'] ) {
			$charge = self::trigger_recurring_payment( compact( 'customer', 'entry', 'action', 'amount' ) );
		} else {
			$charge                   = self::trigger_one_time_payment( compact( 'customer', 'form', 'entry', 'action', 'amount' ) );
			$response['run_triggers'] = true;
		}

		if ( $charge === true ) {
			$response['success'] = true;
		} else {
			$response['error'] = $charge;
		}

		return $response;
	}

	/**
	 * @return bool true if either Stripe Connect or the legacy integration is set up.
	 */
	private static function stripe_is_configured() {
		return FrmStrpAppHelper::call_stripe_helper_class( 'initialize_api' );
	}

	/**
	 * stripeToken: Set when the card will be processed after entry. May not result in successful payment.
	 * stripeMethod: The payment method has been verified.
	 * frmauth: The payment intent has been created.
	 *
	 * @since 2.0
	 */
	private static function has_stripe_processed( $form_id ) {
		return isset( $_POST['stripeToken'] ) || isset( $_POST[ 'frmauth' . $form_id ] ) || isset( $_POST['stripeMethod'] );
	}

	private static function set_customer_with_token( $atts ) {
		if ( isset( $_POST['customer'] ) ) {
			// it's an object if this isn't the first Stripe action running
			return $_POST['customer'];
		}

		$payment_info = array(
			'user_id' => FrmTransAppHelper::get_user_id_for_current_payment(),
		);

		if ( isset( $_POST['stripeToken'] ) ) {
			$payment_info['source'] = sanitize_text_field( $_POST['stripeToken'] );
		}

		if ( ! empty( $atts['action']->post_content['email'] ) ) {
			$payment_info['email'] = apply_filters( 'frm_content', $atts['action']->post_content['email'], $atts['form'], $atts['entry'] );
			if ( $payment_info['email'] === '[email]' ) {
				$payment_info['email'] = FrmProAppHelper::get_current_user_value( 'user_email' );
			}
		}

		self::add_customer_name( $atts, $payment_info );

		$customer          = FrmStrpAppHelper::call_stripe_helper_class( 'get_customer', $payment_info );
		$_POST['customer'] = $customer; // set for later use

		return $customer;
	}

	/**
	 * @since 2.02
	 */
	private static function add_customer_name( $atts, &$payment_info ) {
		if ( empty( $atts['action']->post_content['billing_first_name'] ) ) {
			return;
		}

		$name = '[' . $atts['action']->post_content['billing_first_name'] . ' show="first"]';
		if ( ! empty( $atts['action']->post_content['billing_last_name'] ) ) {
			$name .= ' [' . $atts['action']->post_content['billing_last_name'] . ' show="last"]';
		}

		$payment_info['name'] = apply_filters( 'frm_content', $name, $atts['form'], $atts['entry'] );
	}

	/**
	 * @param array $atts Includes 'customer', 'entry', 'action', 'amount'.
	 * @return bool|string True on success, error message on failure
	 */
	private static function trigger_recurring_payment( $atts ) {
		$new_charge = array(
			'customer'         => $atts['customer']->id,
			'plan'             => self::get_plan_from_atts( $atts ),
			'payment_behavior' => 'allow_incomplete',
			'expand'           => array( 'latest_invoice.payment_intent' ),
			'off_session'      => true,
		);

		$trial_end = self::get_trial_end_time( $atts );
		if ( $trial_end ) {
			$new_charge['trial_end'] = $trial_end;
		}

		$charge                               = FrmStrpAppHelper::call_stripe_helper_class( 'create_subscription', $new_charge );
		$atts['charge']                       = new stdClass();
		$atts['charge']->sub_id               = $charge->id;
		$atts['charge']->id                   = null;
		$atts['charge']->paid                 = 0;
		$atts['charge']->amount               = $atts['amount'];
		$atts['charge']->current_period_start = $charge->current_period_start;
		$atts['charge']->current_period_end   = $charge->current_period_end;
		$atts['charge']->sub_id               = self::create_new_subscription( $atts );
		$atts['status']                       = 'pending';
		self::adjust_for_free_trial( $atts );

		if ( ! empty( $charge->latest_invoice->payment_intent ) ) {
			$intent         = $charge->latest_invoice->payment_intent;
			$payment_status = $intent->status;

			if ( $charge->status === 'incomplete' && ( $payment_status === 'requires_action' || $payment_status === 'requires_source_action' ) ) {
				$confirm_url = FrmStrpAuth::redirect_auth( $atts, $intent->id );
				if ( $confirm_url ) {
					$atts['meta_value'] = array( 'confirm' => $confirm_url );
				}
			}

			$atts['charge']->id = $intent->id;
		}

		self::create_new_payment( $atts );

		$success = true;

		return $success;
	}

	private static function adjust_for_free_trial( &$atts ) {
		$trial_end = self::get_trial_end_time( $atts );
		if ( $trial_end ) {
			$atts['charge']->amount = 0;
			$atts['charge']->current_period_start = time();
			$atts['charge']->current_period_end   = $trial_end;
		}
	}

	/**
	 * Get the trial period from the settings or from the connected entry
	 *
	 * @since 1.16
	 * @param array $atts Includes 'customer', 'entry', 'action', 'amount'.
	 * @return int The timestamp when the trial should end. 0 for no trial
	 */
	private static function get_trial_end_time( $atts ) {
		$settings = $atts['action']->post_content;
		if ( ! empty( $settings['trial_interval_count'] ) ) {
			$trial = $settings['trial_interval_count'];
			if ( ! is_numeric( $trial ) ) {
				$trial = FrmTransAppHelper::process_shortcodes(
					array(
						'value' => $trial,
						'entry' => $atts['entry'],
					)
				);
			}
			if ( $trial ) {
				return strtotime( '+' . absint( $trial ) . ' days' );
			}
		}
		return 0;
	}

	/**
	 * @param array $atts
	 * @return string|true string on error, true on success.
	 */
	private static function trigger_one_time_payment( $atts ) {
		$new_charge = self::new_one_time_array( $atts );

		if ( FrmStrpAuth::process_payment_before() ) {
			$charge = self::capture_one_time_payment( $atts, $new_charge );
		} else {
			$charge = self::run_new_charge( $atts, $new_charge );
		}

		if ( ! is_object( $charge ) ) {
			return $charge;
		}

		if ( isset( $charge->object ) && $charge->object === 'payment_intent' ) {
			// Allow for SCA after entry.
			if ( $charge->status === 'succeeded' ) {
				$atts['status'] = 'complete';
			} elseif ( $charge->status === 'requires_action' || $charge->status === 'requires_source_action' ) {
				$confirm_url = FrmStrpAuth::redirect_auth( $atts, $charge->id );
				if ( $confirm_url ) {
					$atts['meta_value'] = array( 'confirm' => $confirm_url );
				}
				$atts['status'] = 'pending';
			} elseif ( isset( $new_charge['capture'] ) && ! $new_charge['capture'] ) {
				// Don't capture now.
				$atts['status'] = $charge->status === 'requires_capture' ? 'authorized' : 'failed';
			} else {
				// No more authentication required, so capture the charge.
				$payment        = FrmStrpAppHelper::call_stripe_helper_class( 'capture_intent', $charge->id, array( 'amount_to_capture' => $new_charge['amount'] ) );
				$atts['status'] = is_object( $payment ) && 'succeeded' === $payment->status ? 'complete' : 'failed';
			}
		} else {
			$atts['status'] = $charge->paid ? 'complete' : 'failed';
			if ( $charge->paid && ! $charge->captured ) {
				$atts['status'] = 'authorized';
			}
		}

		$atts['charge']         = new stdClass();
		$atts['charge']->id     = $charge->id;
		$atts['charge']->amount = $atts['amount'];

		self::create_new_payment( $atts );

		$success = true;
		return $success;
	}

	/**
	 * @since 2.0
	 * @param array $atts
	 * @return array
	 */
	private static function new_one_time_array( $atts ) {
		$new_charge = array(
			'customer' => $atts['customer']->id,
			'currency' => $atts['action']->post_content['currency'],
			'amount'   => $atts['amount'],
		);
		if ( ! empty( $atts['action']->post_content['description'] ) ) {
			$new_charge['description'] = $atts['action']->post_content['description'];
		}

		// Include 'capture' = false to authorize only.
		if ( isset( $atts['action']->post_content['capture'] ) && $atts['action']->post_content['capture'] === 'authorize' ) {
			$new_charge['capture'] = false;
		}

		return apply_filters( 'frm_strp_new_charge', $new_charge, $atts );
	}

	/**
	 * @since 2.0
	 * @param array $atts
	 * @param array $new_charge
	 * @return mixed
	 */
	private static function capture_one_time_payment( $atts, $new_charge ) {
		$intents = FrmStrpAuth::get_payment_intents( 'frmauth' . $atts['entry']->form_id );
		if ( ! $intents ) {
			return 'Payment authorization not found';
		}

		foreach ( $intents as $intent_id ) {
			$intent = FrmStrpAppHelper::call_stripe_helper_class( 'get_intent', $intent_id );
			if ( $intent->metadata->action != $atts['action']->ID ) {
				continue;
			}

			// Add the description on the payment intent.
			$missing_desc = empty( $intent->description ) && ! empty( $new_charge['description'] );
			$missing_cust = empty( $intent->customer ) && ! empty( $new_charge['customer'] );
			if ( $missing_desc || $missing_cust ) {
				// TODO: include address and name as needed
				$data = array();

				if ( ! empty( $new_charge['description'] ) ) {
					$data['description'] = $new_charge['description'];
				}

				if ( ! empty( $new_charge['customer'] ) ) {
					$data['customer'] = $new_charge['customer'];
				}

				FrmStrpAppHelper::call_stripe_helper_class( 'update_intent', $intent->id, $data );
			}

			$authorize_only = isset( $new_charge['capture'] ) && ! $new_charge['capture'];
			if ( $authorize_only ) {
				// Don't continue if the payment shouldn't be captured.
				$charge           = new stdClass();
				$charge->paid     = true;
				$charge->captured = false;
				$charge->amount   = $new_charge['amount'];
				$charge->id       = $intent_id;
				return $charge;
			}

			$payment = FrmStrpAppHelper::call_stripe_helper_class( 'capture_intent', $intent_id, array( 'amount_to_capture' => $new_charge['amount'] ) );

			if ( ! is_object( $payment ) ) {
				return 'There was an issue trying to capture the payment.';
			}

			$charge = reset( $payment->charges->data );
			return $charge;
		}
	}

	/**
	 * @since 2.0
	 * @param array $atts
	 * @param array $new_charge
	 */
	private static function run_new_charge( $atts, $new_charge ) {
		if ( isset( $_POST['stripeMethod'] ) ) {
			$new_charge['payment_method'] = FrmAppHelper::get_post_param( 'stripeMethod', '', 'sanitize_text_field' );
			$new_charge['confirm']        = true;
			if ( isset( $new_charge['capture'] ) ) {
				// Allow for capture later.
				$new_charge['capture_method'] = $new_charge['capture'] ? 'automatic' : 'manual';
				unset( $new_charge['capture'] );
			}
		}
		return FrmStrpAppHelper::call_stripe_helper_class( 'run_new_charge', $new_charge );
	}

	/**
	 * @param object $e exception.
	 * @return string
	 */
	private static function get_generic_exception( $e ) {
		FrmTransLog::log_message( $e->getMessage() );
		return $e->getMessage();
	}

	private static function create_new_subscription( $atts ) {
		$atts['charge'] = (object) $atts['charge'];

		$new_values = array(
			'amount'         => number_format( ( $atts['charge']->amount / 100 ), 2, '.', '' ),
			'paysys'         => 'stripe',
			'item_id'        => $atts['entry']->id,
			'action_id'      => $atts['action']->ID,
			'sub_id'         => isset( $atts['charge']->sub_id ) ? $atts['charge']->sub_id : '',
			'interval_count' => $atts['action']->post_content['interval_count'],
			'time_interval'  => $atts['action']->post_content['interval'],
			'status'         => 'active',
			'next_bill_date' => date( 'Y-m-d' ),
		);

		$frm_sub = new FrmTransSubscription();
		$sub_id  = $frm_sub->create( $new_values );
		return $sub_id;
	}

	private static function create_new_payment( $atts ) {
		$atts['charge'] = (object) $atts['charge'];

		$new_values = array(
			'amount'     => number_format( ( $atts['charge']->amount / 100 ), 2, '.', '' ),
			'status'     => $atts['status'],
			'paysys'     => 'stripe',
			'item_id'    => $atts['entry']->id,
			'action_id'  => $atts['action']->ID,
			'receipt_id' => $atts['charge']->id,
			'sub_id'     => isset( $atts['charge']->sub_id ) ? $atts['charge']->sub_id : '',
		);

		if ( isset( $atts['charge']->current_period_end ) ) {
			$new_values['begin_date']  = date( 'Y-m-d', $atts['charge']->current_period_start );
			$new_values['expire_date'] = date( 'Y-m-d', $atts['charge']->current_period_end );
		}

		if ( isset( $atts['meta_value'] ) ) {
			$new_values['meta_value'] = $atts['meta_value'];
		}

		$frm_payment = new FrmTransPayment();
		$payment_id  = $frm_payment->create( $new_values );
		return $payment_id;
	}

	/**
	 * Convert the amount from 10.00 to 1000
	 */
	public static function prepare_amount( $amount, $atts = array() ) {
		$amount   = parent::prepare_amount( $amount, $atts );
		$currency = self::get_currency_for_action( $atts );

		return number_format( $amount, $currency['decimals'], '', '' );
	}

	public static function add_action_defaults( $defaults ) {
		$defaults['plan_id'] = '';
		$defaults['capture'] = '';
		return $defaults;
	}

	public static function add_action_options( $atts ) {
		$form_action    = $atts['form_action'];
		$action_control = $atts['action_control'];
		include FrmStrpAppHelper::plugin_path() . '/views/action-settings/options.php';
	}

	public static function before_save_settings( $settings ) {
		$settings['currency'] = strtolower( $settings['currency'] );
		$settings             = self::create_plans( $settings );
		return $settings;
	}

	public static function create_plans( $settings ) {
		if ( $settings['type'] !== 'recurring' || strpos( $settings['amount'], ']' ) ) {
			$settings['plan_id'] = '';
			return $settings;
		}

		$plan_opts = self::prepare_plan_options( $settings );
		if ( $plan_opts['id'] != $settings['plan_id'] ) {
			$settings['plan_id'] = self::maybe_create_plan( $plan_opts );
		}

		return $settings;
	}

	private static function prepare_plan_options( $settings ) {
		$amount              = self::prepare_amount( $settings['amount'], $settings );
		$default_description = number_format( ( $amount / 100 ), 2 ) . '/' . $settings['interval'];
		$plan_opts           = array(
			'amount'         => $amount,
			'interval'       => $settings['interval'],
			'interval_count' => $settings['interval_count'],
			'currency'       => $settings['currency'],
			'name'           => empty( $settings['description'] ) ? $default_description : $settings['description'],
		);

		if ( ! empty( $settings['trial_interval_count'] ) ) {
			$plan_opts['trial_period_days'] = self::get_trial_with_default( $settings['trial_interval_count'] );
		}

		$plan_opts['id'] = self::create_plan_id( $settings );

		return $plan_opts;
	}

	/**
	 * Since the trial period can come from an entry, use a default value
	 * when creating the plan. This is overridden when the subscription
	 * is created.
	 *
	 * @since 1.16
	 */
	private static function get_trial_with_default( $trial ) {
		if ( ! is_numeric( $trial ) ) {
			$trial = 1;
		}
		return absint( $trial );
	}

	private static function get_plan_from_atts( $atts ) {
		$action                         = $atts['action'];
		$action->post_content['amount'] = $atts['amount'];
		return self::get_plan_for_action( $action );
	}

	private static function get_plan_for_action( $action ) {
		$plan_id = $action->post_content['plan_id'];
		if ( ! $plan_id ) {
			// the amount has already been formatted, so add the decimal back in
			$amount                         = $action->post_content['amount'];
			$action->post_content['amount'] = number_format( ( $amount / 100 ), 2, '.', '' );
			$plan_opts                      = self::prepare_plan_options( $action->post_content );
			$plan_id                        = self::maybe_create_plan( $plan_opts );
		}
		return $plan_id;
	}

	/**
	 * @param array $plan
	 * @return mixed
	 */
	private static function maybe_create_plan( $plan ) {
		FrmStrpAppHelper::call_stripe_helper_class( 'initialize_api' );
		return FrmStrpAppHelper::call_stripe_helper_class( 'maybe_create_plan', $plan );
	}

	/**
	 * Include settings in the plan description in order to make sure
	 * the correct plan is used.
	 *
	 * @param array $settings
	 * @return string
	 */
	public static function create_plan_id( $settings ) {
		$amount = self::prepare_amount( $settings['amount'], $settings );
		$id     = sanitize_title_with_dashes( $settings['description'] ) . '_' . $amount . '_' . $settings['interval_count'] . $settings['interval'];
		return $id;
	}

	/**
	 * If this form submits with ajax, load the scripts on the first page
	 *
	 * @param array $params
	 */
	public static function maybe_load_scripts( $params ) {
		if ( $params['form_id'] == $params['posted_form_id'] ) {
			// this form has already been posted, so we aren't on the first page
			return;
		}

		$form = FrmForm::getOne( $params['form_id'] );
		if ( isset( $form->options['ajax_submit'] ) && $form->options['ajax_submit'] ) {
			$credit_card_field = FrmField::getAll(
				array(
					'fi.form_id' => $form->id,
					'type'       => 'credit_card',
				)
			);
			if ( $credit_card_field ) {
				self::load_scripts( $params );
			}
		}
	}

	public static function load_scripts( $params ) {
		if ( FrmAppHelper::is_admin_page( 'formidable-entries' ) ) {
			return;
		}

		if ( wp_script_is( 'formidable-stripe', 'enqueued' ) ) {
			return;
		}

		$settings                = new FrmStrpSettings();
		$stripe_connect_is_setup = FrmStrpConnectHelper::stripe_connect_is_setup();
		$publishable             = $settings->get_active_publishable_key();

		wp_register_script( 'stripe', 'https://js.stripe.com/v3/', array(), '3.0', false );
		wp_enqueue_script( 'formidable-stripe', FrmStrpAppHelper::plugin_url() . '/js/frmstrp' . FrmAppHelper::js_suffix() . '.js', array( 'stripe', 'formidable' ), FrmStrpVersionCheck::get_version(), false );

		if ( ! empty( $params['form_id'] ) ) {
			$action_settings = self::prepare_settings_for_js( $params['form_id'] );
		} else {
			$action_settings   = array();
			$params['form_id'] = 0;
		}

		$ajax_url = admin_url( 'admin-ajax.php', is_ssl() ? 'admin' : 'http' );
		$ajax_url = apply_filters( 'frm_ajax_url', $ajax_url );

		$stripe_vars = array(
			'publishable_key' => $publishable,
			'process'         => $settings->settings->process,
			'form_id'         => $params['form_id'],
			'nonce'           => wp_create_nonce( 'frm_strp_ajax' ),
			'root'            => esc_url_raw( rest_url() ),
			'ajax'            => esc_url_raw( $ajax_url ),
			'api_nonce'       => wp_create_nonce( 'wp_rest' ),
			'settings'        => $action_settings,
			'locale'          => self::get_locale(),
			'style'           => self::prepare_styling( $params['form_id'] ),
		);
		if ( $stripe_connect_is_setup ) {
			$stripe_vars['account_id'] = FrmStrpConnectHelper::get_account_id();
		}
		wp_localize_script( 'formidable-stripe', 'frm_stripe_vars', $stripe_vars );
	}

	/**
	 * Translate styling settings to Stripe.
	 *
	 * @since 2.0
	 */
	private static function prepare_styling( $form_id ) {
		if ( empty( $form_id ) ) {
			return array();
		}

		$style = FrmStylesController::get_form_style( $form_id );
		if ( empty( $style ) ) {
			return array();
		}

		$settings = FrmStylesHelper::get_settings_for_output( $style );
		$disallowed = array( ';', ':', '!important' );
		foreach ( $settings as $k => $s ) {
			if ( is_string( $s ) ) {
				$settings[ $k ] = str_replace( $disallowed, '', $s );
			}
		}

		return array(
			'base' => array(
				'color'         => $settings['text_color'],
				'fontFamily'    => $settings['font'],
				'fontSize'      => $settings['field_font_size'],
				'::placeholder' => array(
					'color'     => $settings['text_color_disabled'],
				),
			),
			'invalid' => array(
				'color'     => $settings['required_color'],
				'iconColor' => $settings['required_color'],
			),
		);
	}

	/**
	 * Set the language for Stripe elements.
	 *
	 * @since 2.0
	 */
	private static function get_locale() {
		$allowed = array( 'ar', 'da', 'de', 'en', 'es', 'fi', 'fr', 'he', 'it', 'ja', 'nl', 'no', 'pl', 'ru', 'sv', 'zh' );
		$current = get_locale();
		$parts   = explode( '_', $current );
		$part    = strtolower( $parts[0] );
		return in_array( $part, $allowed, true ) ? $part : 'auto';
	}
}
