<?php

class FrmStrpAuth {

	/**
	 * If returning from Stripe to authorize a payment, show the message.
	 *
	 * @since 2.0
	 */
	public static function maybe_show_message( $form ) {
		$intent_id = FrmAppHelper::simple_get( 'payment_intent', '', 'sanitize_text_field' );
		if ( ! isset( $_GET['frmstrp'] ) || ! $intent_id ) {
			return $form;
		}

		$frm_payment = new FrmTransPayment();
		$payment     = $frm_payment->get_one_by( $intent_id, 'receipt_id' );
		if ( ! $payment ) {
			return $form;
		}

		$entry_id = FrmAppHelper::simple_get( 'frmstrp', '', 'absint' );
		if ( $entry_id !== $payment->item_id || ! FrmStrpAppHelper::stripe_is_configured() ) {
			return $form;
		}

		// Check if intent was complete.
		$intent = FrmStrpAppHelper::call_stripe_helper_class( 'get_intent', $intent_id );

		// If failed, show last error message.
		$failed = array( 'requires_source', 'requires_payment_method', 'canceled' );
		if ( in_array( $intent->status, $failed, true ) ) {
			$message = '<div class="frm_error_style">' . $intent->last_payment_error->message . '</div>';
			self::insert_error_message( $message, $form );
			return $form;
		}

		$atts = array(
			'entry' => FrmEntry::getOne( $entry_id ),
		);
		self::prepare_success_atts( $atts );

		$atts['fields'] = FrmFieldsHelper::get_form_fields( $atts['form']->id );

		ob_start();
		FrmFormsController::run_success_action( $atts );
		$message = ob_get_contents();
		ob_end_clean();

		return $message;
	}

	/**
	 * Add the parameters the receiving functions are expecting.
	 *
	 * @since 2.0
	 */
	private static function prepare_success_atts( &$atts ) {
		$atts['form']     = FrmForm::getOne( $atts['entry']->form_id );
		$atts['entry_id'] = $atts['entry']->id;

		$opt = 'success_action';
		$atts['conf_method'] = ( isset( $atts['form']->options[ $opt ] ) && ! empty( $atts['form']->options[ $opt ] ) ) ? $atts['form']->options[ $opt ] : 'message';
	}

	/**
	 * Insert a message/error where the form styling will be applied.
	 *
	 * @since 2.0
	 */
	private static function insert_error_message( $message, &$form ) {
		$add_after = '<fieldset>';
		$pos = strpos( $form, $add_after );
		if ( $pos !== false ) {
			$form = substr_replace( $form, $add_after . $message, $pos, strlen( $add_after ) );
		}
	}

	/**
	 * Include the token if going between pages.
	 *
	 * @param object $form The form being submitted.
	 */
	public static function add_hidden_token_field( $form ) {
		$posted_form = FrmAppHelper::get_param( 'form_id', 0, 'post', 'absint' );
		if ( $posted_form != $form->id || FrmFormsController::just_created_entry( $form->id ) ) {
			// Check to make sure the correct form was submitted.
			// Was an entry already created and the form should be loaded fresh?

			$intents = self::maybe_create_intents( $form->id );
			self::include_intents_in_form( $intents, $form );

			return;
		}

		if ( isset( $_POST['stripeToken'] ) ) {
			echo '<input type="hidden" name="stripeToken" value="' . esc_attr( wp_unslash( $_POST['stripeToken'] ) ) . '"/>';
			return;
		}

		if ( isset( $_POST['stripeMethod'] ) ) {
			echo '<input type="hidden" name="stripeMethod" value="' . esc_attr( wp_unslash( $_POST['stripeMethod'] ) ) . '"/>';
			return;
		}

		$auths = self::get_payment_intents( 'frmauth' . $form->id );
		foreach ( $auths as $auth ) {
			echo '<input type="hidden" name="frmauth' . esc_attr( $form->id ) . '[]" value="' . esc_attr( $auth ) . '" />';
		}

		$intents = self::get_payment_intents( 'frmintent' . $form->id );
		if ( ! empty( $intents ) ) {
			self::update_intent_pricing( $form->id, $intents );
		} elseif ( empty( $auths ) ) {
			$intents = self::maybe_create_intents( $form->id );
		}

		self::include_intents_in_form( $intents, $form );
	}

	/**
	 * @since 2.02
	 */
	private static function include_intents_in_form( $intents, $form ) {
		foreach ( $intents as $intent ) {
			if ( is_array( $intent ) ) {
				$id = $intent['id'];
				$action = $intent['action'];
			} else {
				$id     = $intent;
				$action = '';
			}

			echo '<input type="hidden" name="frmintent' . esc_attr( $form->id ) . '[]" value="' . esc_attr( $id ) . '" data-action="' . esc_attr( $action ) . '" />';
		}
	}

	/**
	 * @since 2.0
	 */
	public static function get_payment_intents( $name ) {
		if ( ! isset( $_POST[ $name ] ) ) {
			return array();
		}
		$intents = $_POST[ $name ];
		FrmAppHelper::sanitize_value( 'sanitize_text_field', $intents );
		return $intents;
	}

	/**
	 * Update pricing before authorizing.
	 *
	 * @since 2.0
	 */
	public static function update_intent_ajax() {
		check_ajax_referer( 'frm_strp_ajax', 'nonce' );
		$form = json_decode( stripslashes( $_POST['form'] ), true );
		self::format_form_data( $form );

		$form_id = absint( $form['form_id'] );
		$intents = isset( $form[ 'frmintent' . $form_id ] ) ? $form[ 'frmintent' . $form_id ] : array();

		if ( empty( $intents ) ) {
			wp_die();
		}

		if ( ! is_array( $intents ) ) {
			$intents = array( $intents );
		} else {
			foreach ( $intents as $k => $intent ) {
				if ( is_array( $intent ) && isset( $intent[ $k ] ) ) {
					$intents[ $k ] = $intent[ $k ];
				}
			}
		}

		$_POST = $form;
		self::update_intent_pricing( $form_id, $intents );

		wp_die();
	}

	/**
	 * Update pricing on page turn and non-ajax validation.
	 *
	 * @since 2.0
	 * @param int   $form_id
	 * @param array $intents
	 */
	private static function update_intent_pricing( $form_id, &$intents ) {
		if ( ! isset( $_POST['form_id'] ) || absint( $_POST['form_id'] ) != $form_id ) {
			return;
		}

		$actions = FrmStrpActionsController::get_actions_before_submit( $form_id );
		if ( empty( $actions ) || empty( $intents ) ) {
			return;
		}

		$form = FrmForm::getOne( $form_id );

		try {
			if ( ! FrmStrpAppHelper::call_stripe_helper_class( 'initialize_api' ) ) {
				return;
			}
		} catch ( Exception $e ) {
			// Intent was not created.
			return;
		}

		foreach ( $intents as $k => $intent ) {
			$intent_id = explode( '_secret_', $intent )[0];
			$saved     = FrmStrpAppHelper::call_stripe_helper_class( 'get_intent', $intent_id );
			foreach ( $actions as $action ) {
				if ( $saved->metadata->action != $action->ID ) {
					continue;
				}
				$intents[ $k ] = array(
					'id'     => $intent,
					'action' => $action->ID,
				);

				$amount = $action->post_content['amount'];
				if ( strpos( $amount, '[' ) === false ) {
					// The amount is static, so it doesn't need an update.
					continue;
				}

				// Update amount based on field shortcodes.
				$entry  = self::generate_false_entry();
				$amount = FrmStrpActionsController::prepare_amount( $amount, compact( 'form', 'entry', 'action' ) );
				if ( $saved->amount == $amount || $amount == '000' ) {
					continue;
				}

				FrmStrpAppHelper::call_stripe_helper_class( 'update_intent', $intent_id, array( 'amount' => $amount ) );
			}
		}
	}

	/**
	 * Create an entry object with posted values.
	 *
	 * @since 2.0
	 */
	private static function generate_false_entry() {
		$entry          = new stdClass();
		$entry->post_id = 0;
		$entry->id      = 0;
		$entry->metas   = array();
		foreach ( $_POST as $k => $v ) {
			$k = sanitize_text_field( stripslashes( $k ) );
			$v = wp_unslash( $v );

			if ( $k === 'item_meta' ) {
				foreach ( $v as $f => $value ) {
					FrmAppHelper::sanitize_value( 'wp_kses_post', $value );
					$entry->metas[ absint( $f ) ] = $value;
				}
			} else {
				FrmAppHelper::sanitize_value( 'wp_kses_post', $v );
				$entry->{$k} = $v;
			}
		}
		return $entry;
	}

	/**
	 * Reformat the form data in name => value array.
	 *
	 * @since 2.0
	 */
	private static function format_form_data( &$form ) {
		$formatted = array();

		foreach ( $form as $input ) {
			$key = $input['name'];
			if ( isset( $formatted[ $key ] ) ) {
				if ( is_array( $formatted[ $key ] ) ) {
					$formatted[ $key ][] = $input['value'];
				} else {
					$formatted[ $key ] = array( $formatted[ $key ], $input['value'] );
				}
			} else {
				$formatted[ $key ] = $input['value'];
			}
		}

		parse_str( http_build_query( $formatted ), $form );
	}

	/**
	 * @since 2.0
	 */
	private static function maybe_create_intents( $form_id ) {
		$intents = array();
		if ( ! self::process_payment_before() ) {
			// Don't create intents unless we will be using them.
			return $intents;
		}

		if ( ! FrmStrpAppHelper::call_stripe_helper_class( 'initialize_api' ) ) {
			return $intents;
		}

		$actions = FrmStrpActionsController::get_actions_before_submit( $form_id );
		self::add_amount_to_actions( $form_id, $actions );

		foreach ( $actions as $action ) {
			if ( $action->post_content['type'] == 'recurring' ) {
				// Only one-time payments are handled here.
				continue;
			}

			$amount = $action->post_content['amount'];
			if ( $amount == '000' ) {
				$amount = 100; // Create the intent when the form loads.
			}

			$new_charge = array(
				'amount'             => $amount,
				'currency'           => $action->post_content['currency'],
				'capture_method'     => 'manual', // Authorize only and capture after submit.
				'metadata'           => array( 'action' => $action->ID ),
				'setup_future_usage' => 'off_session',
			);
			$intent     = FrmStrpAppHelper::call_stripe_helper_class( 'create_intent', $new_charge );
			if ( is_object( $intent ) ) {
				$intents[] = array(
					'id'     => $intent->client_secret,
					'action' => $action->ID,
				);
			}
		}

		return $intents;
	}

	/**
	 * @since 2.0
	 */
	private static function add_amount_to_actions( $form_id, &$actions ) {
		if ( empty( $actions ) ) {
			return;
		}
		$form = FrmForm::getOne( $form_id );

		foreach ( $actions as $k => $action ) {
			$amount = self::get_amount_before_submit( compact( 'action', 'form' ) );
			$actions[ $k ]->post_content['amount'] = $amount;
		}
	}

	/**
	 * @since 2.0
	 */
	private static function get_amount_before_submit( $atts ) {
		$amount = $atts['action']->post_content['amount'];
		return FrmStrpActionsController::prepare_amount( $atts['action']->post_content['amount'], $atts );
	}

	/**
	 * Should the payment be processed before or after submit?
	 *
	 * @since 2.0
	 *
	 * @return bool
	 */
	public static function process_payment_before() {
		$settings = new FrmStrpSettings();
		return ( $settings->settings->process === 'before' );
	}

	/**
	 * @since 2.0
	 *
	 * @param array $atts Includes 'customer', 'entry', 'action', 'amount'.
	 * @param int   $intent_id
	 */
	public static function redirect_auth( $atts, $intent_id ) {
		$data = array(
			'return_url' => self::return_url( $atts ),
		);
		$a    = FrmStrpAppHelper::call_stripe_helper_class( 'confirm_intent', $intent_id, $data );
		global $frm_strp_redirect_url;
		$confirm_url           = esc_url_raw( $a->next_action->redirect_to_url->url );
		$frm_strp_redirect_url = $confirm_url;

		add_filter( 'frm_redirect_url', 'FrmStrpAuth::set_redirect_url' );
		add_filter( 'frm_success_filter', 'FrmStrpAuth::trigger_redirect' );

		return $confirm_url;
	}

	/**
	 * Triggered by the frm_redirect_url hook.
	 *
	 * @since 2.0
	 *
	 * @return string
	 */
	public static function set_redirect_url( $url ) {
		global $frm_strp_redirect_url;
		if ( $frm_strp_redirect_url ) {
			$url = $frm_strp_redirect_url;
		}
		return $url;
	}

	/**
	 * Triggered by the frm_success_filter hook.
	 *
	 * @since 2.0
	 *
	 * @return string
	 */
	public static function trigger_redirect() {
		return 'redirect';
	}

	/**
	 * @since 2.0
	 *
	 * @param array $atts
	 */
	private static function return_url( $atts ) {
		$atts = array(
			'entry' => $atts['entry'],
		);
		self::prepare_success_atts( $atts );

		if ( $atts['conf_method'] === 'redirect' ) {
			$redirect = self::get_redirect_url( $atts );
		} else {
			$redirect = self::get_message_url( $atts );
		}

		return $redirect;
	}

	/**
	 * If the form should redirect, get the url to redirect to.
	 *
	 * @since 2.0
	 *
	 * @param array $atts
	 */
	private static function get_redirect_url( $atts ) {
		$success_url = trim( $atts['form']->options['success_url'] );
		$success_url = apply_filters( 'frm_content', $success_url, $atts['form'], $atts['entry'] );
		$success_url = do_shortcode( $success_url );

		$atts['id'] = $atts['entry']->id;

		add_filter( 'frm_redirect_url', 'FrmEntriesController::prepare_redirect_url' );
		return apply_filters( 'frm_redirect_url', $success_url, $atts['form'], $atts );
	}

	/**
	 * If the form should should a message, apend it to the success url.
	 *
	 * @since 2.0
	 *
	 * @param array $atts
	 */
	private static function get_message_url( $atts ) {
		$url  = FrmAppHelper::get_server_value( 'HTTP_REFERER' );

		return add_query_arg( array( 'frmstrp' => $atts['entry_id'] ), $url );
	}
}
